/*
 * linux/arch/arm/mach-uniphier/uspin.c
 *
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/pagemap.h>
#include <linux/interrupt.h>
#include <linux/compiler.h>
#include <linux/signal.h>

#include <asm/system.h>
#include <asm/hardirq.h>
#include <asm/delay.h>
#include <asm/mmu_context.h>

#include <mach/exsvc.h>


#define NO_OWNER                 ((pid_t)(-1))


static DEFINE_SPINLOCK(uspin_mutex);
static u32 orig_psr;
static sigset_t orig_sigset;

static DEFINE_SPINLOCK(refcnt_mutex);
static int uspin_refcnt = 0;
static pid_t uspin_owner = NO_OWNER;


void uspin_lock_kernel_irqsave_inner(unsigned long *flags)
{
	spin_lock_irqsave(&uspin_mutex, *flags);
}
EXPORT_SYMBOL(uspin_lock_kernel_irqsave_inner);

void uspin_unlock_kernel_irqrestore_inner(unsigned long *flags)
{
	spin_unlock_irqrestore(&uspin_mutex, *flags);
}
EXPORT_SYMBOL(uspin_unlock_kernel_irqrestore_inner);

static int asmlinkage uspin_lock_user_irqsave(struct pt_regs *frame)
{
	int ret;
	int i;
	sigset_t newset;
	unsigned long saveflags;

	preempt_disable();
	do {
		/* Lock local refcnt_mutex variable. */
		spin_lock_irqsave(&refcnt_mutex, saveflags);

		/*
		 * If entering this function for the first time,
		 * increment reference counter and set owner PID.
		 */
		if (uspin_refcnt == 0) {
			uspin_refcnt = 1;
			uspin_owner = current->pid;
			ret = 0;
		} else {
			if (uspin_owner == current->pid) {
				ret = uspin_refcnt++;
			} else {
				ret = -1;
			}
		}

		/* Unlock local refcnt_mutex variable. */
		spin_unlock_irqrestore(&refcnt_mutex, saveflags);

		if (ret < 0) {
			preempt_enable();
			preempt_disable();
		}
	} while (ret < 0);

	if (ret == 0) {
		/* OK, we can get the uspin */
		/* Check current CPSR of user space */
		if (frame->ARM_cpsr & PSR_I_BIT) {
			printk(KERN_WARNING "uspin: lock: irq disabled??\n");
			//BUG();
		}

		/* Mask all signals */
		for (i = 0; i < _NSIG_WORDS; i++) {
			newset.sig[i] = ~0;
		}

		/* Do we need check the return value?? */
		sigprocmask(SIG_SETMASK, &newset, &orig_sigset);

		/* 
		 * Enter the critical section.
		 *  - Lock the signal handler of process(for NPTL)
		 *  - Lock the uspin
		 *  - Save original CPSR of user mode
		 *  - Set CPSR.I bit of user mode 
		 *    to disable interrupt in User space
		 */
		spin_lock_irq(&current->sighand->siglock);
		spin_lock_irq(&uspin_mutex);

		orig_psr = frame->ARM_cpsr;
		frame->ARM_cpsr |= PSR_I_BIT;
	}

	return 0;
}

static int exsvc_uspin_lock_user_irqsave(void)
{
	volatile struct pt_regs *p;

	p = exsvc_get_regs();
	return uspin_lock_user_irqsave((struct pt_regs *)p);
}

static int asmlinkage uspin_unlock_user_irqrestore(struct pt_regs *frame)
{
	int ret;
	sigset_t save_sigset;
	unsigned long saveflags;

	/* Check the current CPSR of user mode */
	if ((frame->ARM_cpsr & PSR_I_BIT) == 0) {
		BUG();
	}

	spin_lock_irqsave(&refcnt_mutex, saveflags);
	if (uspin_refcnt == 0) {
		/* unlocked */
		ret = 0;
	} else if (uspin_owner == current->pid) {
		/* locked by current process(= recursive lock) */
		ret = uspin_refcnt;
	} else {
		/* locked by other process */
		ret = -1;
	}
	if (ret <= 0) {
		/* no lock or other thread locked */
		BUG();
	}

	/* Decrement lock reference counter */
	ret = --uspin_refcnt;
	if (ret == 0) {
		uspin_owner = NO_OWNER;
	}
	save_sigset = orig_sigset;
	spin_unlock_irqrestore(&refcnt_mutex, saveflags);


	if (ret == 0) {
		/* OK, we can release the uspin */
		/* 
		 * Leave the critical section.
		 *  - Restore CPSR.I bit of user mode 
		 *    to restore interrupt in User space
		 *  - Restore original CPSR of user mode
		 *  - Unlock the uspin
		 *  - Unlock the signal handler of process(for NPTL)
		 */
		// Restore CPSR of User space and unlock l_spin variable
		frame->ARM_cpsr = (frame->ARM_cpsr & ~PSR_I_BIT) | 
			(orig_psr & PSR_I_BIT);

		spin_unlock(&uspin_mutex);
		spin_unlock(&current->sighand->siglock);

		/* Do we need check the return value?? */
		sigprocmask(SIG_SETMASK, &save_sigset, NULL);

		/* exsvc isn't permit the local_irq_enable state */
		local_irq_disable();
	}
	preempt_enable();

	return 0;
}

static int exsvc_uspin_unlock_user_irqrestore(void)
{
	volatile struct pt_regs *p;

	p = exsvc_get_regs();
	return uspin_unlock_user_irqrestore((struct pt_regs *)p);
}

static int __init init_uspin(void)
{
	int result;

	/* Set ex-syscall for uspin_lock/uspin_unlock */
	result = exsvc_set_table(0, 
		(unsigned long)exsvc_uspin_lock_user_irqsave);
	if (result) {
		printk(KERN_WARNING 
			"uspin: cannot register lock function.\n");
		return -ENOMEM;
	}

	result = exsvc_set_table(1, 
		(unsigned long)exsvc_uspin_unlock_user_irqrestore);
	if (result) {
		printk(KERN_WARNING 
			"uspin: cannot register unlock function.\n");
		return -ENOMEM;
	}

	printk(KERN_INFO "uspin: version 0.22 loaded.\n");

	return 0;
}

static void __exit exit_uspin(void)
{
}

module_init(init_uspin);
module_exit(exit_uspin);
